const float INF = 1000000000000.0;
const float M_PI = 3.14159265359;

const float SCENE1_START_T = 11.8;
const float SCENE2_START_T = 13.8;
const float SCENE3_START_T = 22.8;
const float SCENE4_START_T = 30.0;
const float SCENE5_START_T = 34.0;
const float SCENE6_START_T = 42.0;
const float SCENE7_START_T = 60.0;

bool inScene0 = false;
bool inScene1 = false;
bool inScene2 = false;
bool inScene3 = false;
bool inScene4 = false;
bool inScene5 = false;
bool inScene6 = false;
bool inScene7 = false;


/*
Contains useful snippets made by iq (https://iquilezles.org/articles/) licensed under MIT license.
// The MIT License
// Copyright © 2020 Inigo Quilez
// Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions: The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software. THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

mat3 setCamera( in vec3 ro, in vec3 ta, float cr )
{
	vec3 cw = normalize(ta-ro);
	vec3 cp = vec3(sin(cr), cos(cr),0.0);
	vec3 cu = normalize( cross(cw,cp) );
	vec3 cv =          ( cross(cu,cw) );
    return mat3( cu, cv, cw );
}

// AUDIO FUNCTIONS
float getLevel(float x) {
    return texelFetch(iChannel0, ivec2(int(x*512.), 0), 0).r;
}
// The next 2 functions are borrowed from https://www.shadertoy.com/view/7lVBRw
float logX(float x, float a, float c){
   return 1.0 / (exp(-a*(x-c)) + 1.0);
}
float logisticAmpus(float amp){
   float c = 0.8;
   float a = 19.0;
   
   return (logX(amp, a, c) - logX(0.0, a, c)) / (logX(1.0, a, c) - logX(0.0, a, c));
   // clamps range and domain into [0, 1]
}

float getMuzak(float h)
{
    return logisticAmpus(getLevel(2.0*h / 12000.0));
}

// SDF PRIMITIVES AND OPERATIONS by iq

// Basic functions - https://iquilezles.org/articles/distfunctions/
float dot2( in vec2 v ) { return dot(v,v); }
float dot2( in vec3 v ) { return dot(v,v); }
float ndot( in vec2 a, in vec2 b ) { return a.x*b.x - a.y*b.y; }

// Union, Subtraction, Intersection - exact/bound, bound, bound - https://iquilezles.org/articles/distfunctions/
float opUnion( float d1, float d2 ) { return min(d1,d2); }
float opSubtraction( float d1, float d2 ) { return max(-d1,d2); }
float opIntersection( float d1, float d2 ) { return max(d1,d2); }

// Union function that preserves material definition in .y
vec2 opU(vec2 d1, vec2 d2)
{
    return (d1.x<d2.x)?d1:d2;
}

// Smooth union - bound - https://iquilezles.org/articles/distfunctions/
float opSmoothUnion( float d1, float d2, float k ) {
    float h = clamp( 0.5 + 0.5*(d2-d1)/k, 0.0, 1.0 );
    return mix( d2, d1, h ) - k*h*(1.0-h);
}

// Finite Repetition - https://iquilezles.org/articles/distfunctions/
vec3 opRepLim( vec3 p, in float c, in vec3 l)
{
    return p-c*clamp(round(p/c),-l,l);
}

// Pyramid - exact - https://iquilezles.org/articles/distfunctions/
float sdPyramid( vec3 p, float h)
{
  float m2 = h*h + 0.25;
    
  p.xz = abs(p.xz);
  p.xz = (p.z>p.x) ? p.zx : p.xz;
  p.xz -= 0.5;

  vec3 q = vec3( p.z, h*p.y - 0.5*p.x, h*p.x + 0.5*p.y);
   
  float s = max(-q.x,0.0);
  float t = clamp( (q.y-0.5*p.z)/(m2+0.25), 0.0, 1.0 );
    
  float a = m2*(q.x+s)*(q.x+s) + q.y*q.y;
  float b = m2*(q.x+0.5*t)*(q.x+0.5*t) + (q.y-m2*t)*(q.y-m2*t);
    
  float d2 = min(q.y,-q.x*m2-q.y*0.5) > 0.0 ? 0.0 : min(a,b);
    
  return sqrt( (d2+q.z*q.z)/m2 ) * sign(max(q.z,-p.y));
}


// 3D SDFs

// Sphere - exact - https://iquilezles.org/articles/distfunctions/
float sdSphere( vec3 p, float s )
{
  return length(p)-s;
}

// Box - exact - https://iquilezles.org/articles/distfunctions/
float sdBox( vec3 p, vec3 b )
{
  vec3 q = abs(p) - b;
  return length(max(q,0.0)) + min(max(q.x,max(q.y,q.z)),0.0);
}

// Torus - exact - https://iquilezles.org/articles/distfunctions/
float sdTorus( vec3 p, vec2 t )
{
  vec2 q = vec2(length(p.xz)-t.x,p.y);
  return length(q)-t.y;
}

// Plane - exact - https://iquilezles.org/articles/distfunctions/
float sdPlane( vec3 p, vec3 n, float h )
{
  // n must be normalized
  return dot(p,n) + h;
}

// Capsule / Line - exact - https://iquilezles.org/articles/distfunctions/
float sdCapsule( vec3 p, vec3 a, vec3 b, float r )
{
  vec3 pa = p - a, ba = b - a;
  float h = clamp( dot(pa,ba)/dot(ba,ba), 0.0, 1.0 );
  return length( pa - ba*h ) - r;
}

// Capped Cylinder - exact - https://iquilezles.org/articles/distfunctions/
float sdCappedCylinder( vec3 p, float h, float r )
{
  vec2 d = abs(vec2(length(p.xz),p.y)) - vec2(h,r);
  return min(max(d.x,d.y),0.0) + length(max(d,0.0));
}

// Revolution and extrusion from 2D - exact - https://iquilezles.org/articles/distfunctions/
float opExtrusion( in vec3 p, in float d, in float h )
{
    vec2 w = vec2( d, abs(p.y) - h );
    return min(max(w.x,w.y),0.0) + length(max(w,0.0));
}



// 2D SDFs

// Circle - exact - https://iquilezles.org/articles/distfunctions2d/
float sd2Circle( vec2 p, float r )
{
    return length(p) - r;
}

// Box - exact - https://iquilezles.org/articles/distfunctions2d/
float sd2Box( in vec2 p, in vec2 b )
{
    vec2 d = abs(p)-b;
    return length(max(d,0.0)) + min(max(d.x,d.y),0.0);
}

// Isosceles Trapezoid - exact - https://iquilezles.org/articles/distfunctions2d/
float sd2Trapezoid( in vec2 p, in float r1, float r2, float he )
{
    vec2 k1 = vec2(r2,he);
    vec2 k2 = vec2(r2-r1,2.0*he);
    p.x = abs(p.x);
    vec2 ca = vec2(p.x-min(p.x,(p.y<0.0)?r1:r2), abs(p.y)-he);
    vec2 cb = p - k1 + k2*clamp( dot(k1-p,k2)/dot2(k2), 0.0, 1.0 );
    float s = (cb.x<0.0 && ca.y<0.0) ? -1.0 : 1.0;
    return s*sqrt( min(dot2(ca),dot2(cb)) );
}



// ???
vec2 rotate2d(vec2 v, float a) {
	float s = sin(a);
	float c = cos(a);
	mat2 m = mat2(c, -s, s, c);
	return m * v;
}




// CUSTOM SDF PRIMITIVES

const mat2 gearRotMats[3] = mat2[3]( mat2(.89,.45,-.45,.89), mat2(.59,.81,-.81,.59), mat2(.16,.99,-.99,.16) );
float sdGearTooth( vec3 p )
{
    float d;
    
    // Reflect over both x and y axis
    p.xz = abs(p.xz);
    
    vec2 offset = vec2(0, 1.132);
    vec2 p2 = p.xz - offset;
    d = sd2Trapezoid(p2, 0.11, 0.074, 0.14);
    p2 = gearRotMats[0]*p.xz - offset;
    d = min(d, sd2Trapezoid(p2, 0.11, 0.074, 0.14));
    p2 = gearRotMats[1]*p.xz - offset;
    d = min(d, sd2Trapezoid(p2, 0.11, 0.074, 0.14));
    p2 = gearRotMats[2]*p.xz - offset;
    d = min(d, sd2Trapezoid(p2, 0.11, 0.074, 0.14));
    
    return d;
}

float sdTTY( vec3 p_, float distort )
{
    p_ = p_;
    vec3 p = p_;
    
    float d0 = sd2Circle(p.xz, 1.0); // outer circle
    float d1 = sd2Circle(p.xz, 0.70); // inner circle
    float d2 = sd2Box(p.xz, vec2(1.5, 0.13)); // vertical slit
    float d3 = sd2Box(p.xz, vec2(0.24, 0.9)); // horizontal bar
    float dGT = sdGearTooth(p);
    
    float d = opSubtraction(d1, d0);
    d = opUnion(dGT, d);
    d = opSubtraction(d2, d);
    d = opUnion(d3, d);
    d = opExtrusion(p, d, 0.1);
    
    if (distort >= 0.0) {
        d += 0.15*distort*sin(distort*180.*p.x)*sin(distort*120.*p.y)*sin(distort*180.*p.z);
        d += 0.289*distort*sin(distort*78.69*p.x)*sin(distort*69.420*p.y)*sin(distort*78.69*p.z);
    }
    
    return d;
}


float sdTeekkari( vec3 p )
{
    float d, tmp;
    
    float paaThicc = 1.0 + getMuzak(180.0)/2.0;
    float thicc = 1.0 + getMuzak(150.0)/2.0;
    
    // x=asin(t), y=sin(2t)0≤t≤2π,a>0
    float t = fract(2.5*iTime)*2.0*M_PI;
    vec3 pfoo = vec3(0.02*sin(t), 0.01*sin(2.0*t), 0.0);
    t = fract(2.5*iTime + thicc*10.0 )*2.0*M_PI;
    vec3 pfoo2 = vec3(0.02*sin(t), 0.01*sin(2.0*t), 0.0);
    
    vec3 ruumisAla = pfoo+vec3(0.0, 0.3, 0.0);
    vec3 ruumisYla = pfoo2+vec3(0.0, 0.6, 0.0);
    vec3 paaPos = pfoo2+vec3(0.0, 0.7, 0.0);
    vec3 diff = ruumisYla - ruumisAla;
    vec3 diffPaa = paaPos - ruumisAla;
    
    float rot = M_PI/2.0-getMuzak(95.0)/1.5+0.2;
    float s = sin(rot);
    float c = cos(rot);
    ruumisYla = ruumisAla + vec3(diff.x, s*diff.y + c*diff.z, s*diff.z + c*diff.y);
    paaPos = ruumisAla + vec3(diffPaa.x, s*diffPaa.y + c*diffPaa.z, s*diffPaa.z + c*diffPaa.y);
    
    // ruumis
    d = sdCapsule(p, ruumisAla, ruumisYla, 0.05*thicc);

    // PÄÄ
    tmp = sdSphere(p - (ruumisYla+vec3(0.0, 0.1, 0.0)), 0.1*paaThicc);
    d = opSmoothUnion(d, tmp, 0.02);
    //d = min(d, tmp);
    
    // jalka 1
    tmp = sdCapsule(p, ruumisAla + vec3(-0.05, -0.05, 0.0), vec3(-0.1, 0.0, 0.0), 0.03*thicc);
    //d = min(d, tmp);
    d = opSmoothUnion(d, tmp, 0.05);
    //opSmoothUnion(d, tmp, analSmooth);
    tmp = sdCapsule(p, vec3(-0.1, 0.0, 0.05), vec3(-0.1, 0.0, 0.0), 0.03*thicc);
    d = opSmoothUnion(d, tmp, 0.02);
    //d = min(d, tmp);
    
    // jalka 2
    tmp = sdCapsule(p, ruumisAla + vec3(0.05, -0.05, 0.0), vec3(0.1, 0.0, 0.0), 0.03*thicc);
    d = opSmoothUnion(d, tmp, 0.05);
    tmp = sdCapsule(p, vec3(0.1, 0.0, 0.05), vec3(0.1, 0.0, 0.0), 0.03*thicc);
    d = opSmoothUnion(d, tmp, 0.02);
    
    // kasi 1
    tmp = sdCapsule(p, ruumisYla + vec3(-0.05, -0.02, 0.0), ruumisAla + vec3(-0.15, 0.1, 0.05), 0.02*thicc);
    //d = min(d, tmp);
    d = opSmoothUnion(d, tmp, 0.05);
    //tmp = sdCapsule(p, vec3(-0.1, 0.0, 0.05), vec3(-0.1, 0.0, 0.0), 0.03*thicc);
    //d = min(d, tmp);
    // kasi 2
    tmp = sdCapsule(p, ruumisYla + vec3(0.05, -0.02, 0.0), ruumisAla + vec3(0.15, 0.1, 0.05), 0.02*thicc);
    //d = min(d, tmp);
    d = opSmoothUnion(d, tmp, 0.05);
    
    return d;
}

float sdSpeaker( in vec3 p_ )
{
    float tmp;
    p_ = p_/2.-vec3(0.0, 0.2, 0.0);
    
    float d = sdBox(p_, vec3(0.2, 0.8, 0.15));

    vec3 p = vec3(p_.x, -p_.z, p_.y);
    
    tmp = sdPyramid(p*vec3(2.6,1.,5.)-vec3(0.0, -0.16, 3.4), 0.08);
    d = opSubtraction(tmp, d);
    
    tmp = sdCappedCylinder(p-vec3(0.0, -0.16, 0.4), 0.15, 0.15);
    d = opSubtraction(tmp, d);
    
    tmp = sdCappedCylinder(p-vec3(0.0, -0.16, 0.08), 0.15, 0.15);
    d = opSubtraction(tmp, d);
    
    return d;
}
    
float sdSpeakerCones( in vec3 p_ )
{   
    float tmp, d = INF;

    p_ = p_/2.-vec3(0.0, 0.2, 0.0);
    vec3 p = vec3(p_.x, -p_.z, p_.y);
    
    float coneOffset = 0.10;
    float coneSmooth = 0.05;
    
    float amts[3] = float[3](
        getMuzak(60.0)/100.0,
        getMuzak(80.0)/100.0,
        getMuzak(100.0)/50.0
    );
    
    // Alempi
    tmp = sdTorus(p-vec3(0.0, -coneOffset-0.02-2.0*amts[0], 0.08), vec2(0.13, 0.015+amts[0]));
    d  = min(d, tmp);
    
    tmp = sdTorus(p-vec3(0.0, -coneOffset-0.01-2.0*amts[1], 0.08), vec2(0.09, 0.01+amts[1]));
    d = opSmoothUnion(d, tmp, coneSmooth);
    
    tmp = sdTorus(p-vec3(0.0, -coneOffset-2.0*amts[2], 0.08), vec2(0.05, 0.01+amts[2]));
    d = opSmoothUnion(d, tmp, coneSmooth);
    
    tmp = sdSphere(p-vec3(0.0, -coneOffset-2.0*amts[2], 0.08), 0.02+amts[2]);
    d = opSmoothUnion(d, tmp, coneSmooth);
    
    // Ylempi
    tmp = sdTorus(p-vec3(0.0, -coneOffset-0.02-2.0*amts[0], 0.4), vec2(0.13, 0.015+amts[0]));
    d  = min(d, tmp);
    
    tmp = sdTorus(p-vec3(0.0, -coneOffset-0.01-2.0*amts[1], 0.4), vec2(0.09, 0.01+amts[1]));
    d = opSmoothUnion(d, tmp, coneSmooth);
    
    tmp = sdTorus(p-vec3(0.0, -coneOffset-2.0*amts[2], 0.4), vec2(0.05, 0.01+amts[2]));
    d = opSmoothUnion(d, tmp, coneSmooth);
    
    tmp = sdSphere(p-vec3(0.0, -coneOffset-2.0*amts[2], 0.4), 0.02+amts[2]);
    d = opSmoothUnion(d, tmp, coneSmooth);
    
    return d;
}


vec2 scene(vec3 p_)
{
    vec3 p = p_;

    vec2 res = vec2(p.y, 0.0);
    
    
    float tmp;
    vec3 p2;
    
    // left side
    tmp = sdPlane(p, vec3(1.0, 0.0, 0.0), 5.0);
    res = opU(res, vec2(tmp, 2.0));
    
    // right
    tmp = sdPlane(p, vec3(-1.0, 0.0, 0.0), 5.0);
    res = opU(res, vec2(tmp, 3.0));
    
    // back wall
    tmp = sdPlane(p, vec3(0.0, 0.0, 1.0), 5.0);
    res = opU(res, vec2(tmp, 4.0));
    
    // front wall (behind camera)
    tmp = sdPlane(p, vec3(0.0, 0.0, -1.0), 10.0);
    res = opU(res, vec2(tmp, 5.0));
    
    // roof
    tmp = sdPlane(p, vec3(0.0, -1.0, 0.0), 3.0);
    res = opU(res, vec2(tmp, 1.0));
    
    
    p.x = abs(p.x);
    
    if (inScene0 || inScene3 || inScene4 || inScene5 || inScene6) {
        const mat3 rotMatSpkr1 = mat3(0.989, 0.0, 0.149,
            0.0, 1.0, 0.0,
            -0.149, 0.0, 0.989);
        vec3 speakerPos = p*rotMatSpkr1-vec3(0.5, 0.0, -2.0);
        if (sdBox(speakerPos, 2.5*vec3(0.2, 0.8, 0.15)+vec3(0.0, 0.4, 0.0)) < res.x) {
            tmp = sdSpeaker(speakerPos);
            res = opU(res, vec2(tmp, 11.5));
            tmp = sdSpeakerCones(speakerPos);
            res = opU(res, vec2(tmp, 12.5));
        }
        //tmp = sdBox(speakerPos, 2.5*vec3(0.2, 0.8, 0.15)+vec3(0.0, 0.4, 0.0));
        //res = opU(res, vec2(tmp, 12.5));
    }
    
    if (inScene4 || inScene5 || inScene6) {
        // Teekkari
        if (sdBox(p-vec3(0.3, 0.0, 0.0), vec3(0.2,1.0,0.3)) < res.x) {
            tmp = sdTeekkari(p-vec3(0.3, 0.0, 0.0));
            res = opU(res, vec2(tmp, 10.5));
        }
        //tmp = sdBox(p-vec3(0.3, 0.0, 0.0), vec3(0.2,1.0,0.3));
        //res = opU(res, vec2(tmp, 10.5));
    }

    if (inScene1 || inScene2 || inScene3 || inScene5 || inScene6 || inScene7) {
        float rot = .5*iTime + getMuzak(200.0);
        float SMOOTHNESS = inScene1?0.0:getMuzak(100.0)/5.0;
        p2 = p - vec3(2.4, 0.0, 0.0);
        p2.xz = rotate2d(p2.xz, rot);
        if (sdBox(p2, vec3(1.5, 0.55, 1.5)) < res.x) {
            float distort = (inScene2)?SMOOTHNESS:0.0; // too expensive :D
            tmp = sdTTY(p2 - vec3(0.0, 0.15, 0.0), distort) - SMOOTHNESS;
            res = opU(res, vec2(tmp, 13.5));
        }
        //tmp = sdBox(p2, vec3(1.5, 0.55, 1.5));
        //res = opU(res, vec2(tmp, 12.5));
    }
    
    return res;
}

// https://iquilezles.org/articles/normalsSDF/
vec3 calcNormal( in vec3 pos ) // for function f(p)
{
    #define ZERO (min(iFrame,0)) // non-constant zero
    vec3 n = vec3(0.0);
    for( int i=ZERO; i<4; i++ )
    {
        vec3 e = 0.5773*(2.0*vec3((((i+3)>>1)&1),((i>>1)&1),(i&1))-1.0);
        n += e*scene(pos+0.0005*e).x;
    }
    return normalize(n);
}

// https://iquilezles.org/articles/rmshadows/
float softshadow( in vec3 ro, in vec3 rd, float mint, float maxt, float k )
{
    float res = 1.0;
    float ph = 1e20;
    for( float t=mint; t<maxt; )
    {
        float h = scene(ro + rd*t).x;
        if( h<0.001 )
            return 0.0;
        float y = h*h/(2.0*ph);
        float d = sqrt(h*h-y*y);
        res = min( res, k*d/max(0.0,t-y) );
        ph = h;
        t += h;
    }
    return res;
}

vec3 lightPos = vec3(1.0, 2.8, 2.0);


vec2 march( in vec3 ro, in vec3 rd )
{
    float t = 0.0002;
    vec2 res = vec2(-1.0);
    
    for (int i=0; i<512; i++) {
        vec3 p = ro + t*rd;
        vec2 d = scene(p);
        if (abs(d.x) < 0.0001) {
            res = vec2(t, d.y);
            break;
        }
        if (t > 100.0) {
            break;
        }
        
        t += d.x;
    }

    return res;
}

vec3 render( in vec3 ro, in vec3 rd )
{
    vec3 col = vec3(0.0);
    
    // Light colors
    vec3 ambientColor = vec3(.83, .89, .97) * 0.96 * 0.01;
    vec3 lightColor = vec3(1.0, .90, .77) * 2.25 * 0.2;
    vec3 skyColor = vec3(0.0);//vec3(.36, .53, .73);
    vec3 materialColor = vec3(1.0, 0.0, 1.0);
    
    float K = 1.0;
    for (int i=0; i<(inScene7?2:3); i++) {
        vec2 res = march(ro, rd);
        float t = res.x;
        float mat = res.y;

        if (t > -0.5) {

            // We hit something
            vec3 p = ro + t*rd;
            
            float reflectivity = 0.0;

            // MaterialColor
            if (mat < 0.5) { // Floor
                materialColor = vec3(0.1);
                reflectivity = 0.2;
            }
            if (mat >= 0.5 && mat < 1.5) { // roof
                materialColor = inScene7?vec3(0.0):vec3(0.5, 0.5, 0.5);
                reflectivity = 0.5;
            }
            else if (mat >= 1.5 && mat < 2.5) { // left wall
                materialColor = inScene7?vec3(0.0):vec3(0.5, 0.0, 0.0);
                reflectivity = 0.2;
            }
            else if (mat >= 2.5 && mat < 3.5) { // right wall
                materialColor = inScene7?vec3(0.0):vec3(0.0, 0.5, 0.0);
                reflectivity = 0.2;
            }
            else if (mat >= 3.5 && mat < 4.5) { // back wall
                materialColor = inScene7?vec3(0.0):vec3(0.0, 0.0, 0.5);
                reflectivity = 0.2;
            }
            else if (mat >= 4.5 && mat < 5.5) { // front wall
                materialColor = inScene7?vec3(0.0):vec3(0.5, 0.5, 0.5);
                reflectivity = 0.2;
            }
            
            else if (mat >= 10.5 && mat < 11.5) { // Teekkari
                materialColor = vec3(0.25);
                reflectivity = 1.0;
            }
            else if (mat >= 11.5 && mat < 12.5) { // Speaker case
                materialColor = vec3(0.04);
            }
            else if (mat >= 12.5 && mat < 13.5) { // Speaker cones
                materialColor = vec3(0.02);
                reflectivity = 0.3;
            }
            else if (mat >= 13.5 && mat < 14.5) { // TTY logo
                materialColor = vec3(0.01);
                reflectivity = 0.3;
            }

            //vec3 normal = calcNormal(p);
            vec3 normal = (mat<1.5) ? vec3(0.0,1.0,0.0) : calcNormal(p);
            vec3 ref = normalize(reflect(rd, normal));
            vec3 lightDir = normalize(lightPos - p);
            //vec2 refRes = march(p, ref); // march reflected ray

            // reflect once
            //vec3 refColor = (pass < 1)?render(p, ref, pass+1):vec3(0.0);
            float shadow = softshadow(p, lightDir, 0.02, 2.5, 16.0);
            //shadow = 1.0;

            // Lighting calculations
            vec3 ambient = ambientColor * 0.1;

            float diff = max(dot(normal, lightDir), 0.0) * shadow;
            vec3 diffuse = diff * lightColor;

            float spec = pow(max(dot(lightDir, ref), 0.0), 32.0) * shadow;
            vec3 specular = spec * lightColor;

            col += (1.0-reflectivity) * K * (ambient + diffuse + specular) * materialColor;

            K *= reflectivity;
            ro = p;
            rd = ref;
        } else {
            col += K * skyColor;
            break;
        }
    }

    return col;
}

void mainImage( out vec4 fragColor, in vec2 fragCoord )
{
    float time = .5*iTime;
    inScene0 = iTime < SCENE1_START_T;
    inScene1 = SCENE1_START_T <= iTime && iTime <= SCENE2_START_T;
    inScene2 = SCENE2_START_T <= iTime && iTime <= SCENE3_START_T;
    inScene3 = SCENE3_START_T <= iTime && iTime <= SCENE4_START_T;
    inScene4 = SCENE4_START_T <= iTime && iTime <= SCENE5_START_T;
    inScene5 = SCENE5_START_T <= iTime && iTime <= SCENE6_START_T;
    inScene6 = SCENE6_START_T <= iTime && iTime <= SCENE7_START_T;
    inScene7 = SCENE7_START_T <= iTime;
    
    // Camera
    vec3 ro, ta;
    
    if (inScene7) {
        ro = vec3(2.4, 3.0, 0.0);
        ta = vec3(2.41, 1.0, 0.0);
    }
    
    if (inScene6) {
        // Basic view
        ro = vec3(0.8, 1.5, 3.0);
        ta = vec3(0.0, 1.0, 0.0);
    }
    
    if (inScene5) {
        // speaker cone closeup
        ro = vec3(0.8, 0.6, -1.0);
        ta = vec3(0.7, 0.6, -2.0);
    }
    
    if (inScene4) {
        // closeup teekkarit
        ro = vec3(0.0, 0.6, 1.0);
        ta = vec3(0.0, 0.6, -2.0);
    }
    
    if (inScene3) {
        // closeup tty-logo, speakers behind
        ro = vec3(4.0, 0.6, 2.0);
        ta = vec3(-0.7, 0.6, -2.0);
    }
    
    if (inScene1 || inScene2) {
        // closeup tty-logo
        ro = vec3(0.8, 1.5, 0.8);
        ta = vec3(1.9, 0.2, 0.35);
    }
    
    if (inScene0) {
        // rotate around scene
        ro = vec3(4.0*sin(time-4.5), 1.0, -1.0+4.0*cos(time-4.5));
        ta = vec3(0.0, 0.6, -1.0);
    }
    
    mat3 ca = setCamera( ro, ta, 0.0 );
    
    // Generate ray for pixel
    vec2 p = (2.0*fragCoord-iResolution.xy)/iResolution.y;
    vec3 rd = ca * normalize(vec3(p, 1.5+((inScene4||inScene5||inScene6)?getMuzak(100.0)*0.3:0.0)));
    
    vec3 col = render(ro, rd);
    
    // gamma correction
    col = pow( col, vec3(0.4545) );
    
    // Vignette
    float v = distance(p, vec2(0.0))/2.0;
    const float amount = 0.6;
    const float falloff = 0.01;
    v = smoothstep(0.8, falloff*0.799, v * (amount + falloff));
    col *= v;
    
    col *= min(iTime/SCENE2_START_T, 1.0);
    col *= 1.0-smoothstep(SCENE7_START_T, 65.0, iTime);
    //col *= min(max((iTime-SCENE7_START_T),0.0)/10.000, 1.0);

    fragColor = vec4(col, 1.0);
}